<?php

namespace App\Controller\Dashboard\Shared;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Contracts\Translation\TranslatorInterface;
use App\Form\CheckoutType;
use App\Service\AppServices;
use Payum\Core\Request\GetHumanStatus;
use App\Entity\TicketReservation;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Writer\Csv;
use Dompdf\Dompdf;
use Dompdf\Options;
use Symfony\Component\HttpFoundation\UrlHelper;
use Symfony\Component\Asset\Packages;
use Symfony\Component\Routing\RouterInterface;

class OrderController extends Controller
{
    /**
     * @Route("/attendee/checkout", name="dashboard_attendee_checkout")
     * @Route("/pointofsale/checkout", name="dashboard_pointofsale_checkout")
     */
    public function checkout(Request $request, TranslatorInterface $translator, AppServices $services, RouterInterface $router) {

        if ($this->isGranted("ROLE_ATTENDEE")) {
            $paymentGateways = $services->getPaymentGateways(array())->getQuery()->getResult();
            $form = $this->createForm(CheckoutType::class, null, array('validation_groups' => 'attendee'));
        } else {
            $form = $this->createForm(CheckoutType::class, null, array('validation_groups' => 'pos'));
        }

        $form->handleRequest($request);
        $em = $this->getDoctrine()->getManager();
        $order = null;

        if ($form->isSubmitted()) {

            $order = $services->getOrders(array('status' => 0, 'reference' => $form->getData()['orderReference']))->getQuery()->getOneOrNullResult();

            if ($form->isValid()) {

// Recheck order
                if (!$order) {
                    $this->addFlash('error', $translator->trans('The order can not be found'));
                    return $this->redirectToRoute("dashboard_index");
                }

// Recheck order elements
                if (!count($order->getOrderelements())) {
                    $this->addFlash('error', $translator->trans('You order is empty'));
                    return $this->redirectToRoute("dashboard_index");
                }
                foreach ($order->getOrderelements() as $orderelement) {
// Check event sale status
                    if (!$orderelement->getEventticket()->isOnSale()) {
                        $services->handleCanceledPayment($order->getReference(), $translator->trans('Your order has been automatically canceled because one or more events are no longer on sale'));
                        $this->addFlash('notice', $translator->trans('Your order has been automatically canceled because one or more events are no longer on sale'));
                        return $this->redirectToRoute("dashboard_index");
                    }
// Check event quotas
                    if ($orderelement->getEventticket()->getTicketsLeftCount(true, $this->getUser()) > 0 && $orderelement->getQuantity() > $orderelement->getEventticket()->getTicketsLeftCount(true, $this->getUser())) {
                        $services->handleCanceledPayment($order->getReference(), $translator->trans('Your order has been automatically canceled because one or more event\'s quotas has changed'));
                        $this->addFlash('notice', $translator->trans('Your order has been automatically canceled because one or more event\'s quotas has changed'));
                        return $this->redirectToRoute("dashboard_index");
                    }
// Check ticket reservations
                    foreach ($orderelement->getTicketsReservations() as $ticketReservation) {
                        if ($ticketReservation->isExpired()) {
                            $services->handleCanceledPayment($order->getReference(), $translator->trans('Your order has been automatically canceled because your ticket reservations has been released'));
                            $this->addFlash('notice', $translator->trans('Your order has been automatically canceled because your ticket reservations has been released'));
                            return $this->redirectToRoute("dashboard_index");
                        }
                    }
                    // Check seats
                    if ($orderelement->getEventticket()->getEventdate()->getHasSeatingPlan()) {
                        foreach ($orderelement->getReservedSeats() as $reservedSeat) {
                            if ($orderelement->getEventticket()->isSeatAlreadyReservedByIds($reservedSeat['sectionId'], $reservedSeat['rowId'], $reservedSeat['seatNumber'])) {
                                $services->handleCanceledPayment($order->getReference(), $translator->trans('Your order has been automatically canceled because your ticket reservations has been released'));
                                $this->addFlash('notice', $translator->trans('Your order has been automatically canceled because one or more seats were already reserved'));
                                return $this->redirectToRoute("dashboard_index");
                            }
                        }
                    }
                }

                $storage = $this->get('payum')->getStorage('App\Entity\Payment');

                $orderTotalAmount = $order->getOrderElementsPriceSum(true);

                if ($orderTotalAmount == 0) {
                    $paymentGateway = $em->getRepository("App\Entity\PaymentGateway")->findOneBySlug("free");
                    $gatewayFactoryName = "offline";
                } elseif ($this->isGranted("ROLE_ATTENDEE")) {
                    if (count($paymentGateways) == 0) {
                        $this->addFlash('error', $translator->trans('No payment gateways are currently enabled'));
                        return $this->redirectToRoute("dashboard_attendee_cart");
                    }
                    $gatewayFactoryName = $request->request->get('payment_gateway');
                    $paymentGateway = $services->getPaymentGateways(array('gatewayFactoryName' => $gatewayFactoryName))->getQuery()->getOneOrNullResult();
                } else {
                    $paymentGateway = $em->getRepository("App\Entity\PaymentGateway")->findOneBySlug("point-of-sale");
                    $gatewayFactoryName = "offline";
                }
                if (!$paymentGateway) {
                    $this->addFlash('error', $translator->trans('The payment gateway can not be found'));
                    return $this->redirectToRoute("dashboard_index");
                }

// Sets the choosen payment gateway
                $order->setPaymentGateway($paymentGateway);
                $em->persist($order);

                if ($order->getPayment()) {
                    $payment = $order->getPayment();
                } else {
                    $payment = $storage->create();
                }
// Sets the amount to be paid
                $orderamount = intval(bcmul($orderTotalAmount, 100));

                $payment->setOrder($order);
                $payment->setNumber($services->generateReference(20));
                $payment->setCurrencyCode($services->getSetting("currency_ccy"));
                $payment->setTotalAmount($orderamount); // 1.23 USD = 123
                $payment->setDescription($translator->trans("Payment of tickets purchased on %website_name%", array('%website_name%' => $services->getSetting("website_name"))));
                $payment->setClientId($this->getUser()->getId());
                if ($form->getData()['firstName'] != null && $form->getData()['firstName'] != "") {
                    $payment->setFirstName($form->getData()['firstName']);
                }
                if ($form->getData()['lastName'] != null && $form->getData()['lastName'] != "") {
                    $payment->setLastName($form->getData()['lastName']);
                }
                if ($form->getData()['churchName'] != null && $form->getData()['churchName'] != "") {
                    $payment->setChurchName($form->getData()['churchName']);
                }
                if ($this->isGranted("ROLE_ATTENDEE")) {
                    $payment->setClientEmail($form->getData()['email']);
                    $payment->setCountry($form->getData()['country']);
                    $payment->setState($form->getData()['state']);
                    $payment->setCity($form->getData()['city']);
                    $payment->setPostalcode($form->getData()['postalcode']);
                    $payment->setStreet($form->getData()['street']);
                    $payment->setStreet2($form->getData()['street2']);
                }

                $storage->update($payment);
                $order->setPayment($payment);
                $order->setFirstName($form->getData()['firstName']);
$order->setLastName($form->getData()['lastName']);
$order->setChurchName($form->getData()['churchName']); // This can be null, that's fine
                $em->flush();

                if ($this->isGranted("ROLE_ATTENDEE")) {
                    if ($request->request->get('payment_gateway') == "offline") {
                        $this->addFlash('success', $translator->trans('Your order has been successfully placed, please proceed to the payment as explained in the instructions'));
                        return $this->redirectToRoute("dashboard_attendee_order_details", ['reference' => $payment->getOrder()->getReference()]);
                    } else {
                        if ($request->request->get('payment_gateway') == 'flutterwave') {
                            return $this->redirectToRoute('dashboard_attendee_checkout_flutterwave_redirect_to_payment_url', ['orderReference' => $payment->getOrder()->getReference()]);
                        } elseif ($request->request->get('payment_gateway') == 'mercadopago') {
                            return $this->redirectToRoute('dashboard_attendee_checkout_mercadopago_create_preference', ['orderReference' => $payment->getOrder()->getReference()]);
                        } else {
                            $captureToken = $this->get('payum')->getTokenFactory()->createCaptureToken(
                                    $gatewayFactoryName, $payment, 'dashboard_attendee_checkout_done'
                            );
                        }
                    }
                } else {
                    $captureToken = $this->get('payum')->getTokenFactory()->createCaptureToken(
                            $gatewayFactoryName, $payment, 'dashboard_pointofsale_checkout_done'
                    );
                }

                return $this->redirect($captureToken->getTargetUrl());
            } else {
                $this->addFlash('error', $translator->trans('The form contains invalid data'));

                if ($this->isGranted("ROLE_ATTENDEE")) {
                    return $this->render('Dashboard/Attendee/Order/checkout.html.twig', [
                                'form' => $form->createView(),
                                'paymentGateways' => $paymentGateways,
                                'order' => $order
                    ]);
                } else {
                    return $this->render('Dashboard/PointOfSale/Order/checkout.html.twig', [
                                'form' => $form->createView(),
                                'order' => $order
                    ]);
                }
            }
        } else {

            if (!$request->query->get('orderReference')) {
// Check referer
                $referer = $request->headers->get('referer');
                if (!\is_string($referer) || !$referer) {
                    $this->addFlash('info', $translator->trans('You must review your cart before proceeding to checkout'));
                    return $this->redirectToRoute("dashboard_index");
                }
                if ($this->isGranted("ROLE_ATTENDEE")) {
                    if ($router->match(Request::create($referer)->getPathInfo())['_route'] != "dashboard_attendee_cart") {
                        $this->addFlash('info', $translator->trans('You must review your cart before proceeding to checkout'));
                        return $this->redirectToRoute("dashboard_index");
                    }
                }

// Recheck cart status
                if (!count($this->getUser()->getCartelements())) {
                    $this->addFlash('error', $translator->trans('Your cart is empty'));
                    return $this->redirectToRoute("dashboard_index");
                }

// Check event sale status
                foreach ($this->getUser()->getCartelements() as $cartelement) {
                    if (!$cartelement->getEventticket()->isOnSale()) {
                        $em->remove($cartelement);
                        $em->flush();
                        $this->addFlash('notice', $translator->trans('Your cart has been automatically updated because one or more events are no longer on sale'));
                        return $this->redirectToRoute("dashboard_index");
                    }
                    if ($cartelement->getEventticket()->getTicketsLeftCount() > 0 && $cartelement->getQuantity() > $cartelement->getEventticket()->getTicketsLeftCount()) {
                        $cartelement->setQuantity($cartelement->getEventticket()->getTicketsLeftCount());
                        $em->persist($cartelement);
                        $em->flush();
                        $this->addFlash('notice', $translator->trans('Your cart has been automatically updated because one or more event\'s quotas has changed'));
                        return $this->redirectToRoute("dashboard_index");
                    }
                }

// Remove previous ticket reservations
                if (count($this->getUser()->getTicketreservations())) {
                    foreach ($this->getUser()->getTicketreservations() as $ticketreservation) {
                        $em->remove($ticketreservation);
                    }
                    $em->flush();
                }

                $order = $services->transformCartIntoOrder($this->getUser());
                if (!$order) {
                    $this->addFlash('error', $translator->trans('The order can not be found'));
                    return $this->redirectToRoute("dashboard_index");
                }
                $em->persist($order);
                $em->flush();
                $services->emptyCart($this->getUser());

// Create new ticket reservations according to current cart
                foreach ($order->getOrderelements() as $orderElement) {
                    $ticketreservation = new TicketReservation();
                    $ticketreservation->setEventticket($orderElement->getEventticket());
                    $ticketreservation->setUser($this->getUser());
                    $ticketreservation->setOrderelement($orderElement);
                    $ticketreservation->setQuantity($orderElement->getQuantity());
                    $expiresAt = new \DateTime;
                    $ticketreservation->setExpiresAt($expiresAt->add(new \DateInterval('PT' . $services->getSetting("checkout_timeleft") . 'S')));
                    $orderElement->addTicketsReservation($ticketreservation);
                    $em->persist($ticketreservation);
                    $em->flush();
                }
            } else {
                $order = $services->getOrders(array('status' => 0, 'reference' => $request->query->get('orderReference')))->getQuery()->getOneOrNullResult();
                if (!$order) {
                    $this->addFlash('error', $translator->trans('The order can not be found'));
                    return $this->redirectToRoute("dashboard_index");
                }
            }
        }

        if ($this->isGranted("ROLE_ATTENDEE")) {
            return $this->render('Dashboard/Attendee/Order/checkout.html.twig', [
                        'form' => $form->createView(),
                        'paymentGateways' => $paymentGateways,
                        'order' => $order
            ]);
        } else {
            return $this->render('Dashboard/PointOfSale/Order/checkout.html.twig', [
                        'form' => $form->createView(),
                        'order' => $order
            ]);
        }
    }

    /**
     * @Route("/attendee/checkout/done", name="dashboard_attendee_checkout_done")
     * @Route("/pointofsale/checkout/done", name="dashboard_pointofsale_checkout_done")
     */
    public function done(Request $request, AppServices $services, TranslatorInterface $translator) {

// Remove ticket reservations
        $em = $this->getDoctrine()->getManager();
        if (count($this->getUser()->getTicketreservations())) {
            foreach ($this->getUser()->getTicketreservations() as $ticketreservation) {
                $em->remove($ticketreservation);
            }
            $em->flush();
        }

        try {
            $token = $this->get('payum')->getHttpRequestVerifier()->verify($request);
            $gateway = $this->get('payum')->getGateway($token->getGatewayName());
        } catch (Exception $e) {
            $this->addFlash('error', $translator->trans('An error has occured while processing your request'));
            return $this->redirectToRoute("dashboard_index");
        }
        $gateway->execute($status = new GetHumanStatus($token));
        $payment = $status->getFirstModel();
        $this->get('payum')->getHttpRequestVerifier()->invalidate($token);

        if ($status->isCaptured() || $status->isAuthorized() || $status->isPending()) {
            $services->handleSuccessfulPayment($payment->getOrder()->getReference());
            if ($payment->getOrder()->getOrderElementsPriceSum() > 0) {
                $this->addFlash('success', $translator->trans('Your payment has been successfully processed'));
            } else {
                $this->addFlash('success', $translator->trans('Your registration has been successfully processed'));
            }
            if ($this->isGranted("ROLE_ATTENDEE")) {
                return $this->redirectToRoute("dashboard_attendee_order_details", ['reference' => $payment->getOrder()->getReference()]);
            } else {
                return $this->redirectToRoute("dashboard_pointofsale_order_details", ['reference' => $payment->getOrder()->getReference()]);
            }
        } elseif ($status->isFailed()) {
            $services->handleFailedPayment($payment->getOrder()->getReference());
            $this->addFlash('error', $translator->trans('Your payment could not be processed at this time'));
            if ($this->isGranted("ROLE_ATTENDEE")) {
                return $this->redirectToRoute("dashboard_attendee_checkout_failure", ["number" => $payment->getNumber()]);
            } else {
                return $this->redirectToRoute("dashboard_pointofsale_index");
            }
        } elseif ($status->isCanceled()) {
            $services->handleCanceledPayment($payment->getOrder()->getReference());
            $this->addFlash('error', $translator->trans('Your payment operation was canceled'));
            if ($this->isGranted("ROLE_ATTENDEE")) {
                return $this->redirectToRoute("dashboard_attendee_orders");
            } else {
                return $this->redirectToRoute("dashboard_pointofsale_index");
            }
        } else {
            return $this->redirectToRoute("dashboard_index");
        }
        if ($this->isGranted("ROLE_ATTENDEE")) {
            return $this->render('Dashboard/Attendee/Order/failure.html.twig', [
                        'status' => $status->getValue(),
                        'paymentdetails' => $payment->getDetails()
            ]);
        } else {
            return $this->redirectToRoute("dashboard_index");
        }
    }

    /**
     * @Route("/attendee/checkout/failure/{number}", name="dashboard_attendee_checkout_failure")
     */
    public function failure($number, Request $request, AppServices $services, TranslatorInterface $translator) {

        $referer = $request->headers->get('referer');
        if (!\is_string($referer) || !$referer || $referer != "dashboard_attendee_checkout_done") {
            return $this->redirectToRoute("dashboard_attendee_orders");
        }

        $payment = $services->getPayments(array("number" => $number))->getQuery()->getOneOrNullResult();
        if (!$payment) {
            $this->addFlash('error', $translator->trans('The payment can not be found'));
            return $this->redirectToRoute("dashboard_attendee_orders");
        }

        return $this->render('Dashboard/Attendee/Order/failure.html.twig', [
                    'paymentdetails' => $payment->getDetails()
        ]);
    }
    /**
     * @Route("/attendee/my-tickets/{reference}", name="dashboard_attendee_order_details")
     * @Route("/pointofsale/my-orders/{reference}", name="dashboard_pointofsale_order_details")
     * @Route("/organizer/recent-orders/{reference}", name="dashboard_organizer_order_details")
     * @Route("/administrator/manage-orders/{reference}", name="dashboard_administrator_order_details")
     */
    public function details($reference, Request $request, TranslatorInterface $translator, AppServices $services)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }

        $status = null;
        if ($order->getStatus() == 1) {
            if ($order->getPaymentGateway()->getGatewayName() == "flutterwave" || $order->getPaymentGateway()->getGatewayName() == "mercadopago") {
                $status["value"] = "Captured";
            } else {
                $gateway = $this->get('payum')->getGateway($order->getPaymentGateway()->getGatewayName());
                $gateway->execute($status = new \Payum\Core\Request\GetHumanStatus($order->getPayment()));
            }
        }

        return $this->render('Dashboard/Shared/Order/details.html.twig', [
            'order' => $order,
            'status' => $status
        ]);
    }

    /**
     * @Route("/print-tickets/{reference}", name="dashboard_tickets_print")
     */
    public function printTickets($reference, Request $request, TranslatorInterface $translator, AppServices $services, UrlHelper $urlHelper, Packages $assetsManager)
    {
        $order = $services->getOrders(['reference' => $reference])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $this->redirectToRoute("dashboard_attendee_orders");
        }

        if ($request->getLocale() == "ar") {
            return $this->redirectToRoute("dashboard_tickets_print", ["reference" => $reference, "_locale" => "en"]);
        }

        $eventDateTicketReference = $request->query->get('event', 'all');

        $pdfOptions = new Options();
        $pdfOptions->set('isRemoteEnabled', true);
        $dompdf = new Dompdf($pdfOptions);
        $html = $this->renderView('Dashboard/Shared/Order/ticket-pdf.html.twig', [
            'order' => $order,
            'eventDateTicketReference' => $eventDateTicketReference,
        ]);

        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();
        $dompdf->stream($order->getReference() . "-" . $translator->trans("tickets"), [
            "Attachment" => false
        ]);
        exit(0);
    }

    /**
     * @Route("/attendee/my-tickets", name="dashboard_attendee_orders")
     * @Route("/pointofsale/my-orders", name="dashboard_pointofsale_orders")
     * @Route("/organizer/manage-orders", name="dashboard_organizer_orders")
     * @Route("/administrator/manage-orders", name="dashboard_administrator_orders")
     */
    public function orders(Request $request, AppServices $services, PaginatorInterface $paginator, AuthorizationCheckerInterface $authChecker, TranslatorInterface $translator)
    {
        $reference = ($request->query->get('reference')) == "" ? "all" : $request->query->get('reference');
        $event = ($request->query->get('event')) == "" ? "all" : $request->query->get('event');
        $eventdate = ($request->query->get('eventdate')) == "" ? "all" : $request->query->get('eventdate');
        $eventticket = ($request->query->get('eventticket')) == "" ? "all" : $request->query->get('eventticket');
        $user = ($request->query->get('user')) == "" ? "all" : $request->query->get('user');
        $organizer = ($request->query->get('organizer')) == "" ? "all" : $request->query->get('organizer');
        $datefrom = ($request->query->get('datefrom')) == "" ? "all" : $request->query->get('datefrom');
        $dateto = ($request->query->get('dateto')) == "" ? "all" : $request->query->get('dateto');
        $status = ($request->query->get('status')) == "" ? "all" : $request->query->get('status');
        $paymentgateway = ($request->query->get('paymentgateway')) == "" ? "all" : $request->query->get('paymentgateway');

        $ordersQuery = $services->getOrders(["reference" => $reference, "event" => $event, "eventdate" => $eventdate, "eventticket" => $eventticket, "user" => $user, "organizer" => $organizer, "datefrom" => $datefrom, "dateto" => $dateto, "status" => $status, "paymentgateway" => $paymentgateway])->getQuery();

        $ordersPagination = $paginator->paginate($ordersQuery, $request->query->getInt('page', 1), 10, ['wrap-queries' => true]);

        return $this->render('Dashboard/Shared/Order/orders.html.twig', [
            'orders' => $ordersPagination,
        ]);
    }

    /**
     * @Route("/administrator/manage-orders/{reference}/cancel", name="dashboard_administrator_order_cancel")
     */
    public function cancel($reference, Request $request, TranslatorInterface $translator, AppServices $services)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }

        if ($order->getDeletedAt()) {
            $this->addFlash('error', $translator->trans('The order has been soft deleted, restore it before canceling it'));
            return $services->redirectToReferer('orders');
        }

        if ($order->getStatus() != 0 && $order->getStatus() != 1) {
            $this->addFlash('error', $translator->trans('The order status must be paid or awaiting payment'));
            return $services->redirectToReferer('orders');
        }

        $services->handleCanceledPayment($order->getReference(), $request->query->get('note'));
        $this->addFlash('error', $translator->trans('The order has been permanently canceled'));

        return $services->redirectToReferer('orders');
    }

    /**
     * @Route("/administrator/manage-orders/{reference}/delete", name="dashboard_administrator_order_delete")
     */
    public function delete($reference, Request $request, TranslatorInterface $translator, AppServices $services)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }
        $em = $this->getDoctrine()->getManager();

        if ($order->getDeletedAt()) {
            $this->addFlash('error', $translator->trans('The order has been permanently deleted'));
        } else {
            $this->addFlash('notice', $translator->trans('The order has been deleted'));
        }

        if ($order->getPayment()) {
            $order->getPayment()->setOrder(null);
            $em->persist($order);
            $em->persist($order->getPayment());
        }

        $em->remove($order);
        $em->flush();

        if ($request->query->get('forceRedirect') == "1") {
            return $this->redirectToRoute("dashboard_administrator_orders");
        }

        return $services->redirectToReferer('orders');
    }

    /**
     * @Route("/administrator/manage-orders/{reference}/restore", name="dashboard_administrator_order_restore")
     */
    public function restore($reference, Request $request, TranslatorInterface $translator, AppServices $services)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }

        $order->setDeletedAt(null);
        foreach ($order->getOrderelements() as $orderelement) {
            $orderelement->setDeletedAt(null);
            foreach ($orderelement->getTickets() as $ticket) {
                $ticket->setDeletedAt(null);
            }
            foreach ($orderelement->getTicketsReservations() as $ticketReservation) {
                $ticketReservation->setDeletedAt(null);
            }
        }

        $em = $this->getDoctrine()->getManager();
        $em->persist($order);
        $em->flush();
        $this->addFlash('success', $translator->trans('The order has been succesfully restored'));

        return $services->redirectToReferer('orders');
    }

    /**
     * @Route("/organizer/recent-orders/{reference}/resend-confirmation-email", name="dashboard_organizer_order_resend_confirmation_email")
     * @Route("/administrator/manage-orders/{reference}/resend-confirmation-email", name="dashboard_administrator_order_resend_confirmation_email")
     */
    public function resendConfirmationEmail($reference, Request $request, TranslatorInterface $translator, AppServices $services)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }
        $services->sendOrderConfirmationEmail($order, $request->query->get('email'));
        $this->addFlash('success', $translator->trans('The confirmation email has been resent to') . ' ' . $request->query->get('email'));
        return $services->redirectToReferer('orders');
    }

    /**
     * @Route("/attendee/my-tickets/{reference}/contact-organizer", name="dashboard_attendee_order_contactOrganizer")
     */
    public function contactOrganizer($reference, Request $request, TranslatorInterface $translator, AppServices $services, \Twig_Environment $templating, \Swift_Mailer $mailer)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }

        $message = $request->request->get('message');
        $emailTo = [];
        foreach ($order->getOrderelements() as $orderElement) {
            $emailTo[] = $orderElement->getEventticket()->getEventdate()->getEvent()->getOrganizer()->getUser()->getEmail();
        }

        $email = (new \Swift_Message($services->getSetting('website_name') . ' - ' . $translator->trans('New message regarding the order') . ' #' . $reference))
            ->setFrom($services->getSetting('no_reply_email'))
            ->setTo($emailTo)
            ->setBody(
                $templating->render('Dashboard/Shared/Order/contact-organizer-email.html.twig', ['order' => $order, 'message' => $message]),
                'text/html'
            );

        $mailer->send($email);

        $this->addFlash('success', $translator->trans('Your message has been successfully sent'));
        return $services->redirectToReferer('orders');
    }

    /**
     * @Route("/organizer/recent-orders/{reference}/contact-attendee", name="dashboard_organizer_order_contactAttendee")
     */
    public function contactAttendee($reference, Request $request, TranslatorInterface $translator, AppServices $services, \Twig_Environment $templating, \Swift_Mailer $mailer)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 'all'])->getQuery()->getOneOrNullResult();
        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }

        $message = $request->request->get('message');

        $email = (new \Swift_Message($services->getSetting('website_name') . ' - ' . $translator->trans('New message regarding the order') . ' #' . $reference))
            ->setFrom($services->getSetting('no_reply_email'))
            ->setTo($order->getUser()->getEmail())
            ->setBody(
                $templating->render('Dashboard/Shared/Order/contact-attendee-email.html.twig', ['order' => $order, 'message' => $message, 'organizer' => $this->getUser()->getOrganizer()]),
                'text/html'
            );

        $mailer->send($email);

        $this->addFlash('success', $translator->trans('Your message has been successfully sent'));
        return $services->redirectToReferer('orders');
    }

    /**
     * @Route("/administrator/manage-orders/{reference}/validate", name="dashboard_administrator_order_validate")
     * @Route("/organizer/recent-orders/{reference}/validate", name="dashboard_organizer_order_validate")
     */
    public function validate($reference, Request $request, TranslatorInterface $translator, AppServices $services)
    {
        $order = $services->getOrders(['reference' => $reference, 'status' => 0])->getQuery()->getOneOrNullResult();

        if (!$order) {
            $this->addFlash('error', $translator->trans('The order can not be found'));
            return $services->redirectToReferer('orders');
        }

        if ($order->getDeletedAt()) {
            $this->addFlash('error', $translator->trans('The order has been soft deleted, restore it before canceling it'));
            return $services->redirectToReferer('orders');
        }

        $services->handleSuccessfulPayment($order->getReference());
        $this->addFlash('success', $translator->trans('The order has been successfully validated'));

        return $services->redirectToReferer('orders');
    }
}